!*DECK DGECO
SUBROUTINE DGECO (A, LDA, N, IPVT, RCOND, Z)
!***BEGIN PROLOGUE  DGECO
!***PURPOSE  Factor a matrix using Gaussian elimination and estimate
!            the condition number of the matrix.
!***LIBRARY   SLATEC (LINPACK)
!***CATEGORY  D2A1
!***TYPE      DOUBLE PRECISION (SGECO-S, DGECO-D, CGECO-C)
!***KEYWORDS  CONDITION NUMBER, GENERAL MATRIX, LINEAR ALGEBRA, LINPACK,
!             MATRIX FACTORIZATION
!***AUTHOR  Moler, C. B., (U. of New Mexico)
!***DESCRIPTION
!
!     DGECO factors a double precision matrix by Gaussian elimination
!     and estimates the condition of the matrix.
!
!     If  RCOND  is not needed, DGEFA is slightly faster.
!     To solve  A*X = B , follow DGECO by DGESL.
!     To compute  INVERSE(A)*C , follow DGECO by DGESL.
!     To compute  DETERMINANT(A) , follow DGECO by DGEDI.
!     To compute  INVERSE(A) , follow DGECO by DGEDI.
!
!     On Entry
!
!        A       DOUBLE PRECISION(LDA, N)
!                the matrix to be factored.
!
!        LDA     INTEGER
!                the leading dimension of the array  A .
!
!        N       INTEGER
!                the order of the matrix  A .
!
!     On Return
!
!        A       an upper triangular matrix and the multipliers
!                which were used to obtain it.
!                The factorization can be written  A = L*U  where
!                L  is a product of permutation and unit lower
!                triangular matrices and  U  is upper triangular.
!
!        IPVT    INTEGER(N)
!                an INTEGER vector of pivot indices.
!
!        RCOND   DOUBLE PRECISION
!                an estimate of the reciprocal condition of  A .
!                For the system  A*X = B , relative perturbations
!                in  A  and  B  of size  EPSILON  may cause
!                relative perturbations in  X  of size  EPSILON/RCOND .
!                If  RCOND  is so small that the logical expression
!                           1.0 + RCOND .EQ. 1.0
!                is true, then  A  may be singular to working
!                precision.  In particular,  RCOND  is zero  if
!                exact singularity is detected or the estimate
!                underflows.
!
!        Z       DOUBLE PRECISION(N)
!                a work vector whose contents are usually unimportant.
!                If  A  is close to a singular matrix, then  Z  is
!                an approximate null vector in the sense that
!                NORM(A*Z) = RCOND*NORM(A)*NORM(Z) .
!
!***REFERENCES  J. J. Dongarra, J. R. Bunch, C. B. Moler, and G. W.
!                 Stewart, LINPACK Users' Guide, SIAM, 1979.
!***ROUTINES CALLED  DASUM, DAXPY, DDOT, DGEFA, DSCAL
!***REVISION HISTORY  (YYMMDD)
!   780814  DATE WRITTEN
!   890531  Changed all specific intrinsics to generic.  (WRB)
!   890831  Modified array declarations.  (WRB)
!   890831  REVISION DATE from Version 3.2
!   891214  Prologue converted to Version 4.0 format.  (BAB)
!   900326  Removed duplicate information from DESCRIPTION section.
!           (WRB)
!   920501  Reformatted the REFERENCES section.  (WRB)
!***END PROLOGUE  DGECO

INTEGER(4) :: LDA,N
INTEGER(4), dimension(*)  :: IPVT
real(8), dimension(LDA,*) :: A
real(8), dimension(*)     :: Z
real(8) :: RCOND
!
real(8) :: DDOT,EK,T,WK,WKM
real*8  :: ANORM,S,DASUM,SM,YNORM
INTEGER(4) :: INFO,J,K,KB,KP1,L
!
!     COMPUTE 1-NORM OF A
!
!***FIRST EXECUTABLE STATEMENT  DGECO
ANORM = 0.0D0

do J = 1, N
   ANORM = MAX(ANORM,DASUM(N,A(1,J),1))
end do

!     FACTOR

CALL DGEFA(A,LDA,N,IPVT,INFO)

!     RCOND = 1/(NORM(A)*(ESTIMATE OF NORM(INVERSE(A)))) .
!     ESTIMATE = NORM(Z)/NORM(Y) WHERE  A*Z = Y  AND  TRANS(A)*Y = E .
!     TRANS(A)  IS THE TRANSPOSE OF A .  THE COMPONENTS OF  E  ARE
!     CHOSEN TO CAUSE MAXIMUM LOCAL GROWTH IN THE ELEMENTS OF W  WHERE
!     TRANS(U)*W = E .  THE VECTORS ARE FREQUENTLY RESCALED TO AVOID
!     OVERFLOW.
!
!     SOLVE TRANS(U)*W = E

EK = 1.0D0
do J = 1, N
   Z(J) = 0.0D0
end do

DO K = 1, N
   IF (Z(K) .NE. 0.0D0) EK = SIGN(EK,-Z(K))
   if (ABS(EK-Z(K)) .GT. ABS(A(K,K))) then
      S = ABS(A(K,K))/ABS(EK-Z(K))
      CALL DSCAL(N,S,Z,1)
      EK = S*EK
   end if
   WK = EK - Z(K)
   WKM = -EK - Z(K)
   S = ABS(WK)
   SM = ABS(WKM)
   IF (A(K,K) .EQ. 0.0D0) then
	  WK = 1.0D0
      WKM = 1.0D0
   else	
      WK = WK/A(K,K)
      WKM = WKM/A(K,K)
   end if
   KP1 = K + 1
   IF (KP1 .le. N) then
      DO J = KP1, N
         SM = SM + ABS(Z(J)+WKM*A(K,J))
         Z(J) = Z(J) + WK*A(K,J)
         S = S + ABS(Z(J))
      end do
      IF (S .lt. SM) then
         T = WKM - WK
         WK = WKM
         DO J = KP1, N
            Z(J) = Z(J) + T*A(K,J)
         end do
      end if
   end if
   Z(K) = WK
end do !K

S = 1.0D0/DASUM(N,Z,1)
CALL DSCAL(N,S,Z,1)
!
!     SOLVE TRANS(L)*Y = W
!
DO KB = 1, N
   K = N + 1 - KB
   IF (K .LT. N) Z(K) = Z(K) + DDOT(N-K,A(K+1,K),1,Z(K+1),1)
   IF (ABS(Z(K)) .GT. 1.0D0) then
      S = 1.0D0/ABS(Z(K))
      CALL DSCAL(N,S,Z,1)
   end if
   L = IPVT(K)
   T = Z(L)
   Z(L) = Z(K)
   Z(K) = T
end do !KB
S = 1.0D0/DASUM(N,Z,1)
CALL DSCAL(N,S,Z,1)
!
YNORM = 1.0D0
!
!     SOLVE L*V = Y
!
do K = 1, N
   L = IPVT(K)
   T = Z(L)
   Z(L) = Z(K)
   Z(K) = T
   IF (K .LT. N) CALL DAXPY(N-K,T,A(K+1,K),1,Z(K+1),1)
   IF (ABS(Z(K)) .GT. 1.0D0) then
      S = 1.0D0/ABS(Z(K))
      CALL DSCAL(N,S,Z,1)
      YNORM = S*YNORM
   end if
end do
S = 1.0D0/DASUM(N,Z,1)
CALL DSCAL(N,S,Z,1)
YNORM = S*YNORM
!
!     SOLVE  U*Z = V
!
DO KB = 1, N
   K = N + 1 - KB
   IF (ABS(Z(K)) .GT. ABS(A(K,K))) then
      S = ABS(A(K,K))/ABS(Z(K))
      CALL DSCAL(N,S,Z,1)
      YNORM = S*YNORM
   END IF
   IF (A(K,K) .NE. 0.0D0) Z(K) = Z(K)/A(K,K)
   IF (A(K,K) .EQ. 0.0D0) Z(K) = 1.0D0
   T = -Z(K)
   CALL DAXPY(K-1,T,A(1,K),1,Z(1),1)
END DO
!     MAKE ZNORM = 1.0
S = 1.0D0/DASUM(N,Z,1)
CALL DSCAL(N,S,Z,1)
YNORM = S*YNORM
!
IF (ANORM .NE. 0.0D0) RCOND = YNORM/ANORM
IF (ANORM .EQ. 0.0D0) RCOND = 0.0D0
RETURN
END
